LevelUtils = {}

--Create an event-triggered goal object
function LevelUtils.MakeGoal(enabledFunc, triggers, eventFunc, disabledFunc)

	local goalTable = {enabled = false}
	
	goalTable.triggers = triggers
	
	goalTable.OnEnabled = enabledFunc
	goalTable.OnEvent = eventFunc
	goalTable.OnDisabled = disabledFunc
	
	function goalTable.Enable(self)
		if self.enabled or not self.OnEvent then
			return
		end
		self.enabled = true
		
		if self.OnEnabled then
			self:OnEnabled()
		end
		
		for i,v in ipairs(self.triggers) do
			ScriptMgr:SubscribeToEvent(v, self)
		end
	end

	function goalTable.Disable(self)
		if not self.enabled or not self.OnEvent then
			return
		end
		self.enabled = false
		
		if self.OnDisabled then
			self:OnDisabled()
		end
		
		ScriptMgr:UnsubscribeEvent(self)
	end	
	
	return goalTable
end

-- Create timer that's refreshed by hive kills/captures
function LevelUtils.CreateXenocideTimer(p_baseDuration, p_captureIncrement, p_onTimerExpired)

	local captureTrigger
	local timerLeft = p_baseDuration
	local initialBonus = 0
	local capCount = 0
	local function LevelTimerUpdate(p_milliseconds)
		timerLeft = timerLeft - p_milliseconds
		if timerLeft < 0 then
			timerLeft = 0
		end
		
		initialBonus = initialBonus + p_milliseconds
		
		GameWorld:SetEventIndicator(string.format("%d:%02d", timerLeft / 60000, (timerLeft % 60000) / 1000))
		
		if 	timerLeft == 0 and
			not GameWorld:GetIsGameOver() then
			captureTrigger:Disable()
		
			GameWorld:ClearText()
		
			p_onTimerExpired()	
		end
	end
	ScriptMgr:SetUpdateCallback(LevelTimerUpdate)
	
	captureTrigger = LevelUtils.MakeGoal(
		function (self)
		end,
		
		{[[NT_POINT_CAPTURED]]},
		function (self, p_type, p_entId, p_pos, p_other)	
			timerLeft = timerLeft + p_captureIncrement
			
			if capCount == 0 then
    		    timerLeft = timerLeft + initialBonus	
            end
			
			capCount = capCount + 1
			
			GameWorld:MakeEventInfoWave()
		end)
	captureTrigger:Enable()
	
	--Create a return stub object that allows tinkering with the timer
	local returnObj = {}
	
	function returnObj:ChangeCaptureIncrement(p_newIncrement)
	    p_captureIncrement = p_newIncrement
    end
    
    return returnObj
end

--Create UI for Xenocide reward $$$
function LevelUtils.CreateXenocideRewardCounter()

	local lastRewardValue
	local function RewardUpdate(p_milliseconds)
		
		rewardValue = tonumber(GameWorld:GetParam("ContractReward"))
		if not rewardValue then rewardValue = 0 end
		
		rewardValue = GameWorld:GetStat([[ST_XENOCIDE_REWARD]]) + rewardValue

		if rewardValue == lastRewardValue then
			return
		end
		
		rewardValue = math.floor(rewardValue)
		local commaSeparatedValue = string.gsub(rewardValue, "(%d)(%d%d%d)$", "%1,%2", 1)
		while true do
			commaSeparatedValue, found = string.gsub(commaSeparatedValue, "(%d)(%d%d%d),", "%1,%2,", 1)
			if found == 0 then break end
		end	

		GameWorld:SetUIElementText("RewardCounter", "$" .. commaSeparatedValue)		

		lastRewardValue = rewardValue
	end
	ScriptMgr:SetUpdateCallback(RewardUpdate)

end

-- Create a monitor script that fires a function once HP for an entity drops below certain thresholds; option for refreshing the triggers
function LevelUtils.CreateEntityHPMonitor(p_entHandle, p_thresholdTable, p_thresholdFunc, p_pushBack)
    
    local currentThreshold = 1
    ScriptMgr:SetUpdateCallback(
        function (p_milliseconds)
 
            if #p_thresholdTable <= 0 then
                return
            end
 
            local entPtr = p_entHandle:GetPtr()
            local hpFract = -1
            if entPtr then
                hpFract = entPtr:GetHPFraction()
            end
           
            if p_thresholdTable[currentThreshold] and hpFract < p_thresholdTable[currentThreshold] then
                p_thresholdFunc(p_thresholdTable[currentThreshold])
                currentThreshold = currentThreshold + 1
            end
            
            if p_pushBack then
                if p_thresholdTable[currentThreshold-2] ~= nil and hpFract > p_thresholdTable[currentThreshold-2] then
                    currentThreshold = currentThreshold - 1
                end
                
                if hpFract == 1.0 then
                    currentThreshold = 1
                end
            end

        end)

end

-- Create a timer, with custom format and callback
function LevelUtils.CreateTimer(p_baseDuration, p_timerFormat, p_onTimerExpired)

    local timerTable = {enabled = true}
	local timerLeft = p_baseDuration
	
	local function LevelTimerUpdate(p_milliseconds)
		timerLeft = timerLeft - p_milliseconds
		if timerLeft < 0 then
			timerLeft = 0
		end
		
		local timeStr = string.format("%d:%02d", timerLeft / 60000, (timerLeft % 60000) / 1000)
		GameWorld:SetEventIndicator(string.format(p_timerFormat, timeStr))
		
		if 	timerLeft == 0 and
			not GameWorld:GetIsGameOver() then
		
			p_onTimerExpired()
            ScriptMgr:RemoveUpdateCallback(LevelTimerUpdate)	
		end
	end
	ScriptMgr:SetUpdateCallback(LevelTimerUpdate)
	
	function timerTable.Disable(self)
	    if self.enabled then
	        ScriptMgr:RemoveUpdateCallback(LevelTimerUpdate)
	        GameWorld:SetEventIndicator("")
	        
	        self.enabled = false
	    end
	end
	
	function timerTable.ResetTimer(self, p_newTime)
	    timerLeft = p_newTime
    end
	
	function timerTable.GetTimerLeft(self)
	    return timerLeft
    end
	
	return timerTable
end


-- Create a timer, with custom format and callback
function LevelUtils.ShowTimedDialogue(p_text, p_speaker, p_duration)
    if p_duration == nil then
        p_duration = 10000
    end

    local text_hdl = GameWorld:ShowText(p_text, p_speaker)

    ScriptMgr:DoDelayedCall(p_duration,
        function ()
            text_hdl:ClearText()
        end) 
end


--Return multiple results with all the skin filenames
function LevelUtils.GetAllSkinFilenames()
	return [[jungle.lua]], [[caves.lua]], [[hatchery.lua]]
end


--Pick a random choice from the parameters
function LevelUtils.PickRandomChoice(...)
	argIndex = GetRandomRangeInt(1, arg.n)
	return arg[argIndex]
end

-- Make a map generator function that 
function LevelUtils.MapGenFromSVG(p_mapName)

	return 	function()
				MapGeometry:LoadSVG(p_mapName)
				MapEntities:LoadSVG(p_mapName)
			end
	
end

-- Return default random level entity placement values
LevelUtils.MapGenDefaults = 
{
	--Capture Points
	CapPtControlRadius 					= 60.0,
	CapPtDangerRadius 					= 30.0,
	CapPtExclusionRadius 				= 6.0,
	CapPtMinNearDist 					= 21.0,
	CapPtSpaceRequirement 				= 8.0,
	
	--Crates
	CrateSpaceRequirement 				= 5.0,
	
	--Mortars
	MortarMinPtDist 					= 15.0,
	MortarMaxPtDist 					= 40.0,
	MortarExclusionRadius 				= 5.0,	

	--Turrets
	TurretMinPtDist 					= 12.0,
	TurretMaxPtDist 					= 20.0,
	TurretExclusionRadius 				= 5.0,
	
	--Sheild Projectors
	ShieldProjMinPtDist 				= 5.0,
	ShieldProjMaxPtDist 				= 12.0,
	ShieldProjExclusionRadius 			= 10.0,

	--Protohive
	NecronodeMinPtDist 					= 30.0,
	NecronodeMaxPtDist 					= 60.0,
	NecronodeExclusionRadius			= 10.0,			
	
	--Bulwarks
	HardpointMinPtDist 					= 30.0,
	HardpointMaxPtDist 					= 60.0,
	HardpointExclusionRadius 			= 20.0,

	Hardpoint_GuardsMinPtDist 			= 30.0,
	Hardpoint_GuardsMaxPtDist 			= 60.0,
	Hardpoint_GuardsExclusionRadius 	= 20.0,
	
	Hardpoint_MimicsMinPtDist 			= 30.0,
	Hardpoint_MimicsMaxPtDist 			= 60.0,
	Hardpoint_MimicsExclusionRadius 	= 20.0,
	
	--Siege Cannon
	SiegeMinPtDist 						= 40.0,
	SiegeMaxPtDist 						= 60.0,
	SiegeExclusionRadius 				= 20.0,	
}

--Tuning constants for mutations and xenocide game mode
--Timer values are from timing how much extra time it roughly takes to kill a hive
function LevelUtils.mutationStats(p_timer, p_mobilesReward, p_towersReward, p_hivesReward)

    return  {
                timer = p_timer,
                mobilesReward = p_mobilesReward,
                towersReward = p_towersReward,
                hivesReward = p_hivesReward,
            }

end

LevelUtils.mutationWeights = {}

LevelUtils.mutationWeights["ZU_SPAWN_SPITTERS"]	    = LevelUtils.mutationStats(0.0, 2.0, 0.0, 2000.0)
LevelUtils.mutationWeights["ZU_SPAWN_BRUTES"]		= LevelUtils.mutationStats(0.0, 1.0, 0.0, 2000.0)
LevelUtils.mutationWeights["ZU_SPAWN_ENRAGED"]		= LevelUtils.mutationStats(0.0, 1.0, 0.0, 1000.0)

LevelUtils.mutationWeights["ZU_DEF_SPITTERS"]		= LevelUtils.mutationStats(25.0, 3.0, 0.0, 0.0)
LevelUtils.mutationWeights["ZU_DEF_BRUTES"]		    = LevelUtils.mutationStats( 5.0, 1.0, 0.0, 0.0)
LevelUtils.mutationWeights["ZU_DEF_ENRAGED"]		= LevelUtils.mutationStats(10.0, 1.0, 0.0, 0.0)

LevelUtils.mutationWeights["ZU_WALK_SPEED"]		    = LevelUtils.mutationStats(15.0, 0.5, 0.0, 900.0)

LevelUtils.mutationWeights["ZU_SPIKE_POWER"]		= LevelUtils.mutationStats(10.0, 0.0, 60.0, 0.0)
LevelUtils.mutationWeights["ZU_SHARPNESS"]			= LevelUtils.mutationStats(60.0, 0.0, 400.0, 0.0)
LevelUtils.mutationWeights["ZU_BOMBARDMENT"]		= LevelUtils.mutationStats(15.0, 0.0, 100.0, 0.0)

LevelUtils.mutationWeights["ZU_NECROPHAGE"]		    = LevelUtils.mutationStats(20.0, 0.0, 0.0, 1000.0)
LevelUtils.mutationWeights["ZU_HIVE_SPIKES"]		= LevelUtils.mutationStats(10.0, 0.0, 0.0, 600.0)

LevelUtils.mutationWeights["ZU_ZOMBIE_MINES"]		= LevelUtils.mutationStats(15.0, 0.0, 0.0, 1000.0)
LevelUtils.mutationWeights["ZU_IMMUNITY"]			= LevelUtils.mutationStats(17.0, 0.0, 0.0, 1000.0)

LevelUtils.mutationWeights["ZU_SPAWNER_SPECIAL"]	= LevelUtils.mutationStats(20.0, 0.0, 0.0, 1500.0)
LevelUtils.mutationWeights["ZU_TAKEOVER_SPECIAL"]	= LevelUtils.mutationStats(20.0, 0.0, 0.0, 1500.0)
LevelUtils.mutationWeights["ZU_HARDPOINT"]			= LevelUtils.mutationStats(20.0, 0.0, 0.0, 1500.0)
LevelUtils.mutationWeights["ZU_HARDPOINT_MIMIC"]	= LevelUtils.mutationStats(25.0, 0.0, 0.0, 1700.0)
LevelUtils.mutationWeights["ZU_HARDPOINT_GUARDS"]	= LevelUtils.mutationStats(25.0, 0.0, 0.0, 1700.0)


LevelUtils.mutationWeights["ZU_BONE_SHIELD"]		= LevelUtils.mutationStats(10.0, 0.0, 0.0, 600.0)
LevelUtils.mutationWeights["ZU_NECRONODE"]			= LevelUtils.mutationStats(20.0, 0.0, 0.0, 1200.0)
LevelUtils.mutationWeights["ZU_DEATH_BONES"]		= LevelUtils.mutationStats( 5.0, 0.5, 0.0, 0.0)
LevelUtils.mutationWeights["ZU_SANDKING_GUARDIANS"] = LevelUtils.mutationStats(20.0, 0.0, 200.0, 1000.0)
LevelUtils.mutationWeights["ZU_RUST"]				= LevelUtils.mutationStats(10.0, 0.0, 0.0, 600.0)
LevelUtils.mutationWeights["ZU_SLOWING_POISON"]	    = LevelUtils.mutationStats(10.0, 0.0, 60.0, 0.0)

LevelUtils.mutationWeights["ZU_WILD_MUTATION"]		= LevelUtils.mutationStats(0.0, 0.0, 0.0, 0.0)
LevelUtils.mutationWeights["ZU_TOWER_CONSUMPTION"]	= LevelUtils.mutationStats( 5.0, 0.0, 0.0, 300.0)
LevelUtils.mutationWeights["ZU_TOWER_CARAPACE"]	    = LevelUtils.mutationStats(15.0, 0.0, 100.0, 0.0)
LevelUtils.mutationWeights["ZU_INFESTATION"]		= LevelUtils.mutationStats(15.0, 0.0, 0.0, 1000.0)

LevelUtils.mutationWeights["ZU_REVIVAL"]			= LevelUtils.mutationStats(15.0, 0.0, 0.0, 1000.0)
LevelUtils.mutationWeights["ZU_MARTYRDOM"]			= LevelUtils.mutationStats(15.0, 0.0, 0.0, 1000.0)
LevelUtils.mutationWeights["ZU_REGENERATION"]	    = LevelUtils.mutationStats(15.0, 0.0, 0.0, 1000.0)
LevelUtils.mutationWeights["ZU_CLONES"]			    = LevelUtils.mutationStats(20.0, 0.0, 0.0, 1500.0)
LevelUtils.mutationWeights["ZU_MIMICRY"]			= LevelUtils.mutationStats(20.0, 0.0, 0.0, 1500.0)

